<?php
// security.php - Medidas de segurança avançadas para o Gestor DREEYTV

class Security {
    private $pdo;
    private $recaptcha_secret;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
        $this->recaptcha_secret = 'SUA CHAVE RECAPTCHA AQUI'; // Chave secreta do reCAPTCHA v2
    }
    
    // Google reCAPTCHA v2
    public function verifyRecaptcha($response) {
        if (empty($response)) {
            return false;
        }
        
        $url = 'https://www.google.com/recaptcha/api/siteverify';
        $data = [
            'secret' => $this->recaptcha_secret,
            'response' => $response,
            'remoteip' => $_SERVER['REMOTE_ADDR']
        ];
        
        $options = [
            'http' => [
                'header' => "Content-type: application/x-www-form-urlencoded\r\n",
                'method' => 'POST',
                'content' => http_build_query($data)
            ]
        ];
        
        $context = stream_context_create($options);
        $result = file_get_contents($url, false, $context);
        $response = json_decode($result);
        
        return $response->success;
    }
    
    // CSRF Protection
    public function generateCSRFToken() {
        if (empty($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['csrf_token'];
    }
    
    public function verifyCSRFToken($token) {
        return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
    }
    
    // XSS Protection
    public function cleanInput($data) {
        if (is_array($data)) {
            return array_map([$this, 'cleanInput'], $data);
        }
        
        $data = trim($data);
        $data = stripslashes($data);
        $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
        return $data;
    }
    
    // Rate Limiting para login
    public function checkLoginAttempts($username, $max_attempts = 5, $lockout_time = 900) {
        $ip = $_SERVER['REMOTE_ADDR'];
        $key = "login_attempts_{$ip}_{$username}";
        
        if (!isset($_SESSION[$key])) {
            $_SESSION[$key] = ['attempts' => 0, 'last_attempt' => time()];
        }
        
        $attempts = $_SESSION[$key];
        
        // Reset se passou o tempo de lockout
        if (time() - $attempts['last_attempt'] > $lockout_time) {
            $_SESSION[$key] = ['attempts' => 0, 'last_attempt' => time()];
            return true;
        }
        
        if ($attempts['attempts'] >= $max_attempts) {
            return false;
        }
        
        return true;
    }
    
    public function incrementLoginAttempts($username) {
        $ip = $_SERVER['REMOTE_ADDR'];
        $key = "login_attempts_{$ip}_{$username}";
        
        if (!isset($_SESSION[$key])) {
            $_SESSION[$key] = ['attempts' => 1, 'last_attempt' => time()];
        } else {
            $_SESSION[$key]['attempts']++;
            $_SESSION[$key]['last_attempt'] = time();
        }
    }
    
    public function resetLoginAttempts($username) {
        $ip = $_SERVER['REMOTE_ADDR'];
        $key = "login_attempts_{$ip}_{$username}";
        unset($_SESSION[$key]);
    }
    
    // Password Policy
    public function validatePassword($password) {
        $errors = [];
        
        if (strlen($password) < 8) {
            $errors[] = "A senha deve ter pelo menos 8 caracteres";
        }
        
        if (!preg_match('/[A-Z]/', $password)) {
            $errors[] = "A senha deve conter pelo menos uma letra maiúscula";
        }
        
        if (!preg_match('/[a-z]/', $password)) {
            $errors[] = "A senha deve conter pelo menos uma letra minúscula";
        }
        
        if (!preg_match('/[0-9]/', $password)) {
            $errors[] = "A senha deve conter pelo menos um número";
        }
        
        if (!preg_match('/[^A-Za-z0-9]/', $password)) {
            $errors[] = "A senha deve conter pelo menos um caractere especial";
        }
        
        return $errors;
    }
    
    // Session Security
    public function secureSession() {
        ini_set('session.cookie_httponly', 1);
        ini_set('session.cookie_secure', 1);
        ini_set('session.use_strict_mode', 1);
        
        session_regenerate_id(true);
    }
    
    // Input Validation
    public function validateEmail($email) {
        return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
    }
    
    public function validatePhone($phone) {
        $clean_phone = preg_replace('/[^0-9]/', '', $phone);
        return strlen($clean_phone) >= 10 && strlen($clean_phone) <= 11;
    }
    
    public function sanitizeSQL($string) {
        return $this->pdo->quote($string);
    }
}

// Funções globais de segurança
function sanitize_output($buffer) {
    $search = [
        '/\>[^\S ]+/s',
        '/[^\S ]+\</s',
        '/(\s)+/s',
        '/<!--(.|\s)*?-->/'
    ];
    
    $replace = [
        '>',
        '<',
        '\\1',
        ''
    ];
    
    $buffer = preg_replace($search, $replace, $buffer);
    return $buffer;
}

// Iniciar buffer de saída para sanitização
if (!isset($_SESSION['security_initialized'])) {
    ob_start("sanitize_output");
    $_SESSION['security_initialized'] = true;
}
?>